<?php
/**
 * Comprehensive Test for Daily Rebate Payout System
 * Tests all components of the VIP-based rebate system
 */

require_once("../../conn.php");
require_once("rebate_functions.php");

// Set timezone
date_default_timezone_set('Asia/Kolkata');

echo "<h2>Daily Rebate Payout System - Comprehensive Test</h2>";

// Test 1: Database connectivity
echo "<h3>Test 1: Database Connectivity</h3>";
if ($conn->connect_error) {
    echo "<div style='background: #f8d7da; padding: 10px; border-radius: 5px; color: #721c24;'>";
    echo "<strong>FAIL:</strong> Database connection failed: " . $conn->connect_error;
    echo "</div>";
    exit;
}

echo "<div style='background: #d4edda; padding: 10px; border-radius: 5px; color: #155724;'>";
echo "<strong>✓ PASS:</strong> Database connection successful";
echo "</div>";

// Test 2: Table existence
echo "<h3>Test 2: Required Tables</h3>";
$requiredTables = ['bet_rebates', 'vip', 'shonu_kaichila'];
$allTablesExist = true;

foreach ($requiredTables as $table) {
    $tableCheck = $conn->query("SHOW TABLES LIKE '$table'");
    if ($tableCheck->num_rows == 0) {
        echo "<div style='background: #f8d7da; padding: 10px; border-radius: 5px; color: #721c24;'>";
        echo "<strong>FAIL:</strong> Table '$table' does not exist";
        echo "</div>";
        $allTablesExist = false;
    } else {
        echo "<div style='background: #d4edda; padding: 10px; border-radius: 5px; color: #155724;'>";
        echo "<strong>✓ PASS:</strong> Table '$table' exists";
        echo "</div>";
    }
}

if (!$allTablesExist) {
    echo "<div style='background: #f8d7da; padding: 10px; border-radius: 5px; color: #721c24;'>";
    echo "<strong>CRITICAL:</strong> Some required tables are missing. Cannot proceed.";
    echo "</div>";
    exit;
}

// Test 3: Check required columns
echo "<h3>Test 3: Required Columns</h3>";
$requiredColumns = [
    'bet_rebates' => ['is_paid', 'paid_at', 'vip_level'],
    'vip' => ['user_id', 'vip_level'],
    'shonu_kaichila' => ['user_id', 'wallet_balance']
];

$allColumnsExist = true;
foreach ($requiredColumns as $table => $columns) {
    $structureQuery = "DESCRIBE $table";
    $structureResult = $conn->query($structureQuery);
    
    if ($structureResult) {
        $existingColumns = [];
        while ($row = $structureResult->fetch_assoc()) {
            $existingColumns[] = $row['Field'];
        }
        
        foreach ($columns as $column) {
            if (in_array($column, $existingColumns)) {
                echo "<div style='background: #d4edda; padding: 5px; border-radius: 3px; color: #155724; margin: 2px;'>";
                echo "<strong>✓ PASS:</strong> Column '$table.$column' exists";
                echo "</div>";
            } else {
                echo "<div style='background: #f8d7da; padding: 5px; border-radius: 3px; color: #721c24; margin: 2px;'>";
                echo "<strong>FAIL:</strong> Column '$table.$column' missing";
                echo "</div>";
                $allColumnsExist = false;
            }
        }
    }
}

if (!$allColumnsExist) {
    echo "<div style='background: #fff3cd; padding: 10px; border-radius: 5px; color: #856404;'>";
    echo "<strong>WARNING:</strong> Some required columns are missing. Run migrate_rebate_table.php";
    echo "</div>";
}

// Test 4: VIP Rate Functions
echo "<h3>Test 4: VIP Rate Functions</h3>";
$vipRates = [
    0 => 0.0000,  // No VIP
    1 => 0.0005,  // VIP 1
    2 => 0.0008,  // VIP 2
    3 => 0.0012,  // VIP 3
    4 => 0.0015,  // VIP 4
    5 => 0.0018,  // VIP 5
    6 => 0.0020,  // VIP 6
    7 => 0.0022,  // VIP 7
    8 => 0.0025,  // VIP 8
    9 => 0.0028,  // VIP 9
    10 => 0.0030  // VIP 10
];

$allRatesCorrect = true;
foreach ($vipRates as $vipLevel => $expectedRate) {
    $actualRate = getVipRebateRate($vipLevel);
    if (abs($actualRate - $expectedRate) < 0.0001) {
        echo "<div style='background: #d4edda; padding: 3px; border-radius: 3px; color: #155724; margin: 1px;'>";
        echo "<strong>✓ PASS:</strong> VIP $vipLevel rate: " . ($actualRate * 100) . "%";
        echo "</div>";
    } else {
        echo "<div style='background: #f8d7da; padding: 3px; border-radius: 3px; color: #721c24; margin: 1px;'>";
        echo "<strong>FAIL:</strong> VIP $vipLevel rate: Expected " . ($expectedRate * 100) . "%, Got " . ($actualRate * 100) . "%";
        echo "</div>";
        $allRatesCorrect = false;
    }
}

if ($allRatesCorrect) {
    echo "<div style='background: #d4edda; padding: 10px; border-radius: 5px; color: #155724;'>";
    echo "<strong>✓ PASS:</strong> All VIP rate functions work correctly";
    echo "</div>";
}

// Test 5: Sample Data Retrieval
echo "<h3>Test 5: Sample Data Retrieval</h3>";
try {
    // Query to get unpaid rebates for testing
    $testQuery = "SELECT 
                    br.id,
                    br.user_id,
                    br.rebate_amount,
                    br.vip_level,
                    sk.wallet_balance,
                    DATE(br.created_at) as rebate_date
                  FROM bet_rebates br
                  JOIN shonu_kaichila sk ON br.user_id = sk.user_id
                  WHERE (br.is_paid IS NULL OR br.is_paid = 0)
                  AND DATE(br.created_at) = CURDATE() - INTERVAL 1 DAY
                  LIMIT 5";
    
    $testResult = $conn->query($testQuery);
    
    if ($testResult) {
        echo "<div style='background: #d4edda; padding: 10px; border-radius: 5px; color: #155724;'>";
        echo "<strong>✓ PASS:</strong> Query executed successfully";
        echo "</div>";
        
        if ($testResult->num_rows > 0) {
            echo "<h4>Sample Unpaid Rebates:</h4>";
            echo "<table border='1' style='border-collapse: collapse; width: 100%;'>";
            echo "<tr style='background: #f2f2f2;'>";
            echo "<th>ID</th><th>User ID</th><th>Rebate Amount</th><th>VIP Level</th><th>Wallet Balance</th><th>Date</th>";
            echo "</tr>";
            
            while ($row = $testResult->fetch_assoc()) {
                echo "<tr>";
                echo "<td>" . htmlspecialchars($row['id']) . "</td>";
                echo "<td>" . htmlspecialchars($row['user_id']) . "</td>";
                echo "<td>₹" . number_format($row['rebate_amount'], 2) . "</td>";
                echo "<td>" . htmlspecialchars($row['vip_level']) . "</td>";
                echo "<td>₹" . number_format($row['wallet_balance'], 2) . "</td>";
                echo "<td>" . htmlspecialchars($row['rebate_date']) . "</td>";
                echo "</tr>";
            }
            echo "</table>";
        } else {
            echo "<div style='background: #fff3cd; padding: 10px; border-radius: 5px; color: #856404;'>";
            echo "<strong>INFO:</strong> No unpaid rebates found for yesterday";
            echo "</div>";
        }
    } else {
        echo "<div style='background: #f8d7da; padding: 10px; border-radius: 5px; color: #721c24;'>";
        echo "<strong>FAIL:</strong> Query failed: " . $conn->error;
        echo "</div>";
    }
} catch (Exception $e) {
    echo "<div style='background: #f8d7da; padding: 10px; border-radius: 5px; color: #721c24;'>";
    echo "<strong>ERROR:</strong> " . $e->getMessage();
    echo "</div>";
}

// Test 6: Cron Job File Check
echo "<h3>Test 6: Cron Job Files</h3>";
$cronFiles = [
    'daily_rebate_payout_cron.php' => 'Main cron job file',
    'rebate_functions.php' => 'Rebate functions file',
    'migrate_rebate_table.php' => 'Database migration file',
    'setup_rebate_cron.php' => 'Cron setup file'
];

foreach ($cronFiles as $file => $description) {
    if (file_exists($file)) {
        echo "<div style='background: #d4edda; padding: 5px; border-radius: 3px; color: #155724; margin: 2px;'>";
        echo "<strong>✓ PASS:</strong> $description ($file) exists";
        echo "</div>";
    } else {
        echo "<div style='background: #f8d7da; padding: 5px; border-radius: 3px; color: #721c24; margin: 2px;'>";
        echo "<strong>FAIL:</strong> $description ($file) missing";
        echo "</div>";
    }
}

// Test 7: Permissions Check
echo "<h3>Test 7: File Permissions</h3>";
$logFile = 'rebate_cron.log';
$lockFile = 'rebate_cron.lock';

if (is_writable('.')) {
    echo "<div style='background: #d4edda; padding: 10px; border-radius: 5px; color: #155724;'>";
    echo "<strong>✓ PASS:</strong> Directory is writable for log and lock files";
    echo "</div>";
} else {
    echo "<div style='background: #f8d7da; padding: 10px; border-radius: 5px; color: #721c24;'>";
    echo "<strong>FAIL:</strong> Directory is not writable. Cron job may fail.";
    echo "</div>";
}

// Test 8: System Status Summary
echo "<h3>Test 8: System Status Summary</h3>";
$systemStatus = [
    'database' => $conn->connect_error ? false : true,
    'tables' => $allTablesExist,
    'columns' => $allColumnsExist,
    'vip_rates' => $allRatesCorrect,
    'permissions' => is_writable('.')
];

$allPassed = true;
foreach ($systemStatus as $test => $passed) {
    if (!$passed) {
        $allPassed = false;
        break;
    }
}

if ($allPassed) {
    echo "<div style='background: #d4edda; padding: 15px; border-radius: 5px; color: #155724; border: 2px solid #28a745;'>";
    echo "<strong>🎉 ALL TESTS PASSED!</strong><br>";
    echo "Your Daily Rebate Payout System is ready to use!<br>";
    echo "✓ Database connectivity: OK<br>";
    echo "✓ Required tables: OK<br>";
    echo "✓ Required columns: OK<br>";
    echo "✓ VIP rate functions: OK<br>";
    echo "✓ File permissions: OK<br>";
    echo "<br><strong>Next Steps:</strong><br>";
    echo "1. Set up the cron job to run daily at 12:01 AM<br>";
    echo "2. Monitor the rebate_cron.log file for execution logs<br>";
    echo "3. Test with a small batch of users first";
    echo "</div>";
} else {
    echo "<div style='background: #f8d7da; padding: 15px; border-radius: 5px; color: #721c24; border: 2px solid #dc3545;'>";
    echo "<strong>❌ SOME TESTS FAILED!</strong><br>";
    echo "Please fix the issues mentioned above before using the system.<br>";
    echo "<br><strong>Common Solutions:</strong><br>";
    echo "• Run migrate_rebate_table.php to add missing columns<br>";
    echo "• Check database permissions<br>";
    echo "• Verify all required files are uploaded<br>";
    echo "• Contact your system administrator if needed";
    echo "</div>";
}

// Test 9: Recent Activity
echo "<h3>Test 9: Recent Activity</h3>";
$recentQuery = "SELECT 
                  COUNT(*) as total_rebates,
                  SUM(rebate_amount) as total_amount,
                  COUNT(DISTINCT user_id) as unique_users,
                  MAX(created_at) as latest_rebate
                FROM bet_rebates 
                WHERE created_at >= DATE_SUB(NOW(), INTERVAL 7 DAYS)";

$recentResult = $conn->query($recentQuery);
if ($recentResult) {
    $recentData = $recentResult->fetch_assoc();
    echo "<div style='background: #e7f3ff; padding: 10px; border-radius: 5px; color: #004085;'>";
    echo "<strong>📊 Last 7 Days Activity:</strong><br>";
    echo "• Total Rebates: " . number_format($recentData['total_rebates']) . "<br>";
    echo "• Total Amount: ₹" . number_format($recentData['total_amount'], 2) . "<br>";
    echo "• Unique Users: " . number_format($recentData['unique_users']) . "<br>";
    echo "• Latest Rebate: " . htmlspecialchars($recentData['latest_rebate']) . "<br>";
    echo "</div>";
}

$conn->close();
?>

<style>
body {
    font-family: Arial, sans-serif;
    max-width: 1200px;
    margin: 0 auto;
    padding: 20px;
    background-color: #f8f9fa;
}

h2 {
    color: #333;
    border-bottom: 2px solid #007bff;
    padding-bottom: 10px;
}

h3 {
    color: #555;
    margin-top: 30px;
    margin-bottom: 15px;
}

table {
    margin: 10px 0;
    background: white;
    border-radius: 5px;
    overflow: hidden;
    box-shadow: 0 2px 4px rgba(0,0,0,0.1);
}

th {
    background-color: #007bff;
    color: white;
    padding: 12px;
    text-align: left;
}

td {
    padding: 8px 12px;
    border-bottom: 1px solid #eee;
}

tr:hover {
    background-color: #f8f9fa;
}
</style>
